% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_groupChromPeaks-functions.R
\name{do_groupChromPeaks_density}
\alias{do_groupChromPeaks_density}
\title{Core API function for peak density based chromatographic peak
grouping}
\usage{
do_groupChromPeaks_density(
  peaks,
  sampleGroups,
  bw = 30,
  minFraction = 0.5,
  minSamples = 1,
  binSize = 0.25,
  maxFeatures = 50,
  sleep = 0,
  index = seq_len(nrow(peaks)),
  ppm = 0
)
}
\arguments{
\item{peaks}{A \code{matrix} or \code{data.frame} with the mz values and
retention times of the identified chromatographic peaks in all samples
of an experiment. Required columns are \code{"mz"}, \code{"rt"} and
\code{"sample"}. The latter should contain \code{numeric} values representing
the index of the sample in which the peak was found.}

\item{sampleGroups}{For \code{PeakDensityParam}: A vector of the same length than
samples defining the sample group assignments (i.e. which samples
belong to which sample group). This parameter is mandatory for
\code{PeakDensityParam} and has to be defined also if there is no sample
grouping in the experiment (in which case all samples should be
assigned to the same group). Samples for which a \code{NA} is provided will
not be considered in the feature definitions step. Providing \code{NA} for
all blanks in an experiment will for example avoid features to be
defined for signals (chrom peaks) present only in blank samples.}

\item{bw}{For \code{PeakDensityParam}: \code{numeric(1)} defining the bandwidth
(standard deviation ot the smoothing kernel) to be used. This argument
is passed to the [density() method.}

\item{minFraction}{For \code{PeakDensityParam}: \code{numeric(1)} defining the minimum
fraction of samples in at least one sample group in which the peaks
have to be present to be considered as a peak group (feature).}

\item{minSamples}{For \code{PeakDensityParam}: \code{numeric(1)} with the minimum
number of samples in at least one sample group in which the peaks have
to be detected to be considered a peak group (feature).}

\item{binSize}{For \code{PeakDensityParam}: \code{numeric(1)} defining the size of the
overlapping slices in m/z dimension.}

\item{maxFeatures}{For \code{PeakDensityParam}: \code{numeric(1)} with the maximum
number of peak groups to be identified in a single mz slice.}

\item{sleep}{\code{numeric(1)} defining the time to \emph{sleep} between
iterations and plot the result from the current iteration.}

\item{index}{An optional \code{integer} providing the indices of the peaks in the
original peak matrix.}

\item{ppm}{For \code{MzClustParam}: \code{numeric(1)} representing the relative m/z
error for the clustering/grouping (in parts per million).
For \code{PeakDensityParam}: \code{numeric(1)} to define m/z-dependent, increasing
m/z bin sizes. If \code{ppm = 0} (the default) m/z bins are defined by the
sequence of values from the smallest to the larges m/z value with a
constant bin size of \code{binSize}. For \code{ppm} > 0 the size of each bin is
increased in addition by the \code{ppm} of the (upper) m/z boundary of the
bin. The maximal bin size (used for the largest m/z values) would then
be \code{binSize} plus \code{ppm} parts-per-million of the largest m/z value of
all peaks in the data set.}
}
\value{
A \code{data.frame}, each row representing a (mz-rt) feature (i.e. a peak group)
with columns:
\itemize{
\item \code{"mzmed"}: median of the peaks' apex mz values.
\item \code{"mzmin"}: smallest mz value of all peaks' apex within the feature.
\item \code{"mzmax"}:largest mz value of all peaks' apex within the feature.
\item \code{"rtmed"}: the median of the peaks' retention times.
\item \code{"rtmin"}: the smallest retention time of the peaks in the group.
\item \code{"rtmax"}: the largest retention time of the peaks in the group.
\item \code{"npeaks"}: the total number of peaks assigned to the feature.
\item \code{"peakidx"}: a \code{list} with the indices of all peaks in a feature in the
\code{peaks} input matrix.
}

Note that this number can be larger than the total number of samples, since
multiple peaks from the same sample could be assigned to a feature.
}
\description{
The \code{do_groupChromPeaks_density} function performs chromatographic peak
grouping based on the density (distribution) of peaks, found in different
samples, along the retention time axis in slices of overlapping m/z ranges.
By default (with parameter \code{ppm = 0}) these m/z ranges have all the same
(constant) size (depending on parameter \code{binSize}). For values of \code{ppm}
larger than 0 the m/z bins (ranges or slices) will have increasing sizes
depending on the m/z value. This better models the m/z-dependent
measurement error/precision seen on some MS instruments.
}
\details{
For overlapping slices along the mz dimension, the function
calculates the density distribution of identified peaks along the
retention time axis and groups peaks from the same or different samples
that are close to each other. See (Smith 2006) for more details.
}
\note{
The default settings might not be appropriate for all LC/GC-MS setups,
especially the \code{bw} and \code{binSize} parameter should be adjusted
accordingly.
}
\examples{
## Load the test file
library(xcms)
library(MsExperiment)
faahko_sub <- loadXcmsData("faahko_sub2")

## Disable parallel processing for this example
register(SerialParam())

## Extract the matrix with the identified peaks from the xcmsSet:
pks <- chromPeaks(faahko_sub)

## Perform the peak grouping with default settings:
res <- do_groupChromPeaks_density(pks, sampleGroups = rep(1, 3))

## The feature definitions:
head(res)
}
\references{
Colin A. Smith, Elizabeth J. Want, Grace O'Maille, Ruben Abagyan and
Gary Siuzdak. "XCMS: Processing Mass Spectrometry Data for Metabolite
Profiling Using Nonlinear Peak Alignment, Matching, and Identification"
Anal. Chem. 2006, 78:779-787.
}
\seealso{
Other core peak grouping algorithms: 
\code{\link{do_groupChromPeaks_nearest}()},
\code{\link{do_groupPeaks_mzClust}()}
}
\author{
Colin Smith, Johannes Rainer
}
\concept{core peak grouping algorithms}
