% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/do_findChromPeaks-functions.R
\name{do_findChromPeaks_matchedFilter}
\alias{do_findChromPeaks_matchedFilter}
\title{Core API function for matchedFilter peak detection}
\usage{
do_findChromPeaks_matchedFilter(
  mz,
  int,
  scantime,
  valsPerSpect,
  binSize = 0.1,
  impute = "none",
  baseValue,
  distance,
  fwhm = 30,
  sigma = fwhm/2.3548,
  max = 5,
  snthresh = 10,
  steps = 2,
  mzdiff = 0.8 - binSize * steps,
  index = FALSE,
  sleep = 0
)
}
\arguments{
\item{mz}{Numeric vector with the individual m/z values from all scans/
spectra of one file/sample.}

\item{int}{Numeric vector with the individual intensity values from all
scans/spectra of one file/sample.}

\item{scantime}{Numeric vector of length equal to the number of
spectra/scans of the data representing the retention time of each scan.}

\item{valsPerSpect}{Numeric vector with the number of values for each
spectrum.}

\item{binSize}{\code{numeric(1)} specifying the width of the
bins/slices in m/z dimension.}

\item{impute}{Character string specifying the method to be used for missing
value imputation. Allowed values are \code{"none"} (no linear
interpolation), \code{"lin"} (linear interpolation), \code{"linbase"}
(linear interpolation within a certain bin-neighborhood) and
\code{"intlin"}. See \code{\link{imputeLinInterpol}} for more details.}

\item{baseValue}{The base value to which empty elements should be set. This
is only considered for \code{method = "linbase"} and corresponds to the
\code{profBinLinBase}'s \code{baselevel} argument.}

\item{distance}{For \code{method = "linbase"}: number of non-empty
neighboring element of an empty element that should be considered for
linear interpolation. See details section for more information.}

\item{fwhm}{\code{numeric(1)} specifying the full width at half maximum
of matched filtration gaussian model peak. Only used to calculate the
actual sigma, see below.}

\item{sigma}{\code{numeric(1)} specifying the standard deviation (width)
of the matched filtration model peak.}

\item{max}{\code{numeric(1)} representing the maximum number of peaks
that are expected/will be identified per slice.}

\item{snthresh}{\code{numeric(1)} defining the signal to noise ratio cutoff.}

\item{steps}{\code{numeric(1)} defining the number of bins to be
merged before filtration (i.e. the number of neighboring bins that will
be joined to the slice in which filtration and peak detection will be
performed).}

\item{mzdiff}{\code{numeric(1)} representing the minimum difference in m/z
dimension required for peaks with overlapping retention times; can be
negative to allow overlap. During peak post-processing, peaks
defined to be overlapping are reduced to the one peak with the largest
signal.}

\item{index}{\code{logical(1)} specifying whether indicies should be
returned instead of values for m/z and retention times.}

\item{sleep}{\code{numeric(1)} defining the number of seconds to wait between
iterations. Defaults to \code{sleep = 0}. If \code{> 0} a plot is
generated visualizing the identified chromatographic peak. Note: this
argument is for backward compatibility only and will be removed in
future.}
}
\value{
A matrix, each row representing an identified chromatographic peak,
    with columns:
    \describe{
    \item{mz}{Intensity weighted mean of m/z values of the peak across scans.}
    \item{mzmin}{Minimum m/z of the peak.}
    \item{mzmax}{Maximum m/z of the peak.}
    \item{rt}{Retention time of the peak's midpoint.}
    \item{rtmin}{Minimum retention time of the peak.}
    \item{rtmax}{Maximum retention time of the peak.}
    \item{into}{Integrated (original) intensity of the peak.}
    \item{intf}{Integrated intensity of the filtered peak.}
    \item{maxo}{Maximum intensity of the peak.}
    \item{maxf}{Maximum intensity of the filtered peak.}
    \item{i}{Rank of peak in merged EIC (\code{<= max}).}
    \item{sn}{Signal to noise ratio of the peak}
    }
}
\description{
This function identifies peaks in the chromatographic
    time domain as described in [Smith 2006]. The intensity values are
    binned by cutting The LC/MS data into slices (bins) of a mass unit
    (\code{binSize} m/z) wide. Within each bin the maximal intensity is
    selected. The peak detection is then performed in each bin by
    extending it based on the \code{steps} parameter to generate slices
    comprising bins \code{current_bin - steps +1} to
    \code{current_bin + steps - 1}.
    Each of these slices is then filtered with matched filtration using
    a second-derative Gaussian as the model peak shape. After filtration
    peaks are detected using a signal-to-ration cut-off. For more details
    and illustrations see [Smith 2006].
}
\details{
The intensities are binned by the provided m/z values within each
    spectrum (scan). Binning is performed such that the bins are centered
    around the m/z values (i.e. the first bin includes all m/z values between
    \code{min(mz) - bin_size/2} and \code{min(mz) + bin_size/2}).

    For more details on binning and missing value imputation see
    \code{\link{binYonX}} and \code{\link{imputeLinInterpol}} methods.
}
\note{
This function exposes core peak detection functionality of
    the \emph{matchedFilter} method. While this function can be called
    directly, users will generally call the corresponding method for the
    data object instead (e.g. the \code{link{findPeaks.matchedFilter}}
    method).
}
\examples{

## Load the test file
faahko_sub <- loadXcmsData("faahko_sub")

## Subset to one file and restrict to a certain retention time range
data <- filterRt(filterFile(faahko_sub, 1), c(2500, 3000))

## Get m/z and intensity values
mzs <- mz(data)
ints <- intensity(data)

## Define the values per spectrum:
valsPerSpect <- lengths(mzs)

res <- do_findChromPeaks_matchedFilter(mz = unlist(mzs), int = unlist(ints),
    scantime = rtime(data), valsPerSpect = valsPerSpect)
head(res)
}
\references{
Colin A. Smith, Elizabeth J. Want, Grace O'Maille, Ruben Abagyan and
Gary Siuzdak. "XCMS: Processing Mass Spectrometry Data for Metabolite
Profiling Using Nonlinear Peak Alignment, Matching, and Identification"
\emph{Anal. Chem.} 2006, 78:779-787.
}
\seealso{
\code{\link{binYonX}} for a binning function,
    \code{\link{imputeLinInterpol}} for the interpolation of missing values.
    \code{\link{matchedFilter}} for the standard user interface method.

Other core peak detection functions: 
\code{\link{do_findChromPeaks_centWave}()},
\code{\link{do_findChromPeaks_centWaveWithPredIsoROIs}()},
\code{\link{do_findChromPeaks_massifquant}()},
\code{\link{do_findPeaks_MSW}()}
}
\author{
Colin A Smith, Johannes Rainer
}
\concept{core peak detection functions}
