% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DataClasses.R, R/functions-Params.R,
%   R/methods-OnDiskMSnExp.R
\name{findPeaks-MSW}
\alias{findPeaks-MSW}
\alias{MSW}
\alias{MSWParam}
\alias{findChromPeaks,OnDiskMSnExp,MSWParam-method}
\title{Single-spectrum non-chromatography MS data peak detection}
\usage{
MSWParam(
  snthresh = 3,
  verboseColumns = FALSE,
  scales = c(1, seq(2, 30, 2), seq(32, 64, 4)),
  nearbyPeak = TRUE,
  peakScaleRange = 5,
  ampTh = 0.01,
  minNoiseLevel = ampTh/snthresh,
  ridgeLength = 24,
  peakThr = NULL,
  tuneIn = FALSE,
  ...
)

\S4method{findChromPeaks}{OnDiskMSnExp,MSWParam}(
  object,
  param,
  BPPARAM = bpparam(),
  return.type = "XCMSnExp",
  msLevel = 1L,
  ...
)
}
\arguments{
\item{snthresh}{\code{numeric(1)} defining the signal to noise ratio cutoff.}

\item{verboseColumns}{\code{logical(1)} whether additional peak meta data
columns should be returned.}

\item{scales}{Numeric defining the scales of the continuous wavelet
transform (CWT).}

\item{nearbyPeak}{\code{logical(1)} whether to include nearby peaks of
major peaks.}

\item{peakScaleRange}{\code{numeric(1)} defining the scale range of the
peak (larger than 5 by default).}

\item{ampTh}{\code{numeric(1)} defining the minimum required relative
amplitude of the peak (ratio of the maximum of CWT coefficients).}

\item{minNoiseLevel}{\code{numeric(1)} defining the minimum noise level
used in computing the SNR.}

\item{ridgeLength}{\code{numeric(1)} defining the minimum highest scale
of the peak in 2-D CWT coefficient matrix.}

\item{peakThr}{\code{numeric(1)} with the minimum absolute intensity
(above baseline) of peaks to be picked. If provided, the smoothing
Savitzky-Golay filter is used (in the \emph{MassSpecWavelet})
package to estimate the local intensity.}

\item{tuneIn}{\code{logical(1)} whther to tune in the parameter
estimation of the detected peaks.}

\item{...}{Additional parameters to be passed to the
\code{peakDetectionCWT()} and
\code{identifyMajorPeaks()} functions from the
\emph{MassSpecWavelet} package.}

\item{object}{For \code{findChromPeaks()}: an
\code{OnDiskMSnExp} object containing the MS- and all
other experiment-relevant data.

\if{html}{\out{<div class="sourceCode">}}\preformatted{For all other methods: a parameter object.
}\if{html}{\out{</div>}}}

\item{param}{An \code{MSWParam} object containing all settings for
the algorithm.}

\item{BPPARAM}{A parameter class specifying if and how parallel processing
should be performed. It defaults to \code{\link[BiocParallel:register]{BiocParallel::bpparam()}}.
See documentation of the \emph{BiocParallel} package for more details. If
parallel processing is enabled, peak detection is performed in parallel
on several of the input samples.}

\item{return.type}{Character specifying what type of object the method should
return. Can be either \code{"XCMSnExp"} (default), \code{"list"} or
\code{"xcmsSet"}.}

\item{msLevel}{\code{integer(1)} defining the MS level on which the peak
detection should be performed. Defaults to \code{msLevel = 1}.}
}
\value{
The \code{MSWParam()} function returns a \code{MSWParam}
class instance with all of the settings specified for peak detection by
the \emph{MSW} method.

For \code{findChromPeaks()}: if \code{return.type = "XCMSnExp"} an
\code{XCMSnExp} object with the results of the peak detection.
If \code{return.type = "list"} a list of length equal to the number of
samples with matrices specifying the identified peaks.
If \code{return.type = "xcmsSet"} an \code{xcmsSet} object
with the results of the detection.
}
\description{
Perform peak detection in mass spectrometry
direct injection spectrum using a wavelet based algorithm.

The \verb{findChromPeaks,OnDiskMSnExp,MSWParam()}
method performs peak detection in single-spectrum non-chromatography MS
data using functionality from the \emph{MassSpecWavelet} package on all
samples from an \code{OnDiskMSnExp} object.
\code{OnDiskMSnExp} objects encapsule all experiment
specific data and load the spectra data (mz and intensity values) on the
fly from the original files applying also all eventual data
manipulations.
}
\details{
This is a wrapper for the peak picker in Bioconductor's
\emph{MassSpecWavelet} package calling
\code{peakDetectionCWT} and
\code{tuneInPeakInfo} functions. See the
\emph{xcmsDirect} vignette for more information.

Parallel processing (one process per sample) is supported and can
be configured either by the \code{BPPARAM} parameter or by globally
defining the parallel processing mode using the
\code{\link[BiocParallel:register]{BiocParallel::register()}} method from the \emph{BiocParallel}
package.
}
\examples{

library(MSnbase)
## Create a MSWParam object
mp <- MSWParam(snthresh = 15)
mp

## Loading a small subset of direct injection, single spectrum files
library(msdata)
fticrf <- list.files(system.file("fticr-mzML", package = "msdata"),
                    recursive = TRUE, full.names = TRUE)
fticr <- readMSData(fticrf[1], msLevel. = 1, mode = "onDisk")

## Perform the MSW peak detection on these:
p <- MSWParam(scales = c(1, 7), peakThr = 80000, ampTh = 0.005,
             SNR.method = "data.mean", winSize.noise = 500)
fticr <- findChromPeaks(fticr, param = p)

head(chromPeaks(fticr))
}
\seealso{
The \code{\link[=do_findPeaks_MSW]{do_findPeaks_MSW()}} core API function
and \code{\link[=findPeaks.MSW]{findPeaks.MSW()}} for the old user interface.

\code{\link[=XCMSnExp]{XCMSnExp()}} for the object containing the results of
the peak detection.

Other peak detection methods: 
\code{\link{findChromPeaks}()},
\code{\link{findChromPeaks-centWave}},
\code{\link{findChromPeaks-centWaveWithPredIsoROIs}},
\code{\link{findChromPeaks-massifquant}},
\code{\link{findChromPeaks-matchedFilter}}
}
\author{
Joachim Kutzera, Steffen Neumann, Johannes Rainer
}
\concept{peak detection methods}
