% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prep_fiji.R
\name{prep_fiji}
\alias{prep_fiji}
\alias{prep_fiji_image}
\alias{prep_fiji_coords}
\title{Prepare Fiji outputs for building a SpatialExperiment}
\usage{
prep_fiji_image(sample_info, out_dir, lowres_max_size = 1200)

prep_fiji_coords(sample_info, out_dir)
}
\arguments{
\item{sample_info}{A \code{data.frame()} with columns \code{capture_area},
\code{group}, \code{fiji_xml_path}, \code{fiji_image_path},
\code{spaceranger_dir}, \code{intra_group_scalar}, and
\code{group_hires_scalef}. The last two are made by \code{rescale_fiji_inputs()}.}

\item{out_dir}{A \code{character(1)} vector giving a path to a directory to
place the output pixel coordinates CSVs. It must exist in advance.}

\item{lowres_max_size}{An \code{integer(1)} vector: the resolution (number of
pixels) of the larger dimension of the output image(s), considered to be "low
resolution". The default value of \code{1200} assumes that you are stitching
together at most a 2 by 2 grid of Visium capture areas, where each has at
most 600 pixels on the longest dimension (as is the default in SpaceRanger).}
}
\value{
This function returns a \code{character()} with the file paths to the
files it created. For \code{prep_fiji_coords()}, these are the \code{tissue_positions.csv}
files; for \code{prep_fiji_image()}, these are the \code{tissue_lowres_image.png} and
\code{scalefactors_json.json} files.
}
\description{
Together, \code{prep_fiji_image()} and \code{prep_fiji_coords()} process Fiji outputs
and generate one directory per group resembling Spaceranger's
\href{https://www.10xgenomics.com/support/software/space-ranger/latest/analysis/outputs/spatial-outputs}{spatial outputs};
in particular, \code{tissue_positions.csv}, \code{tissue_lowres_image.png}, and
\code{scalefactors_json.json} files are created. These functions are necessary to
run in preparation for \code{build_SpatialExperiment()}.
}
\details{
Given a \code{data.frame()} of sample information (\code{sample_info}) with
columns \code{capture_area}, \code{group}, and \code{fiji_xml_path},
expected to have one unique path to Fiji XML output per group, \code{prep_fiji_coords}
reads in the pixel coordinates from each capture area's \code{tissue_positions.csv}
file from SpaceRanger, and transform using the rotation matrix specified
by Fiji \url{https://imagej.net/software/fiji/}. It writes one new \code{tissue_positions.csv}
file per group.

After stitching all groups in \code{sample_info} with Fiji, images of
various resolutions (pixel dimensions) are left. \code{prep_fiji_image()} creates copies
of each image whose largest dimension is \code{lowres_max_size} pixels. It
also creates a corresponding \code{scalefactors_json.json} file much like
SpaceRanger's.
}
\section{Functions}{
\itemize{
\item \code{prep_fiji_image()}: Create low-res images and scale factors from high-res
Fiji output images

\item \code{prep_fiji_coords()}: Apply transform info from Fiji XML output

}}
\examples{
sample_info <- dplyr::tibble(
    group = "Br2719",
    capture_area = c("V13B23-283_A1", "V13B23-283_C1", "V13B23-283_D1")
)
#   Add 'spaceranger_dir' column
sr_dir <- tempdir()
temp <- unzip(
    spatialLIBD::fetch_data("visiumStitched_brain_spaceranger"),
    exdir = sr_dir
)
sample_info$spaceranger_dir <- file.path(
    sr_dir, sample_info$capture_area, "outs", "spatial"
)

#   Add Fiji-output-related columns
fiji_dir <- tempdir()
temp <- unzip(
    spatialLIBD::fetch_data("visiumStitched_brain_Fiji_out"),
    exdir = fiji_dir
)
sample_info$fiji_xml_path <- temp[grep("xml$", temp)]
sample_info$fiji_image_path <- temp[grep("png$", temp)]

## Re-size images and add more information to the sample_info
sample_info <- rescale_fiji_inputs(sample_info, out_dir = tempdir())

spe_input_dir <- tempdir()
out_paths_image <- prep_fiji_image(
    sample_info,
    out_dir = spe_input_dir, lowres_max_size = 1000
)
out_path_coords <- prep_fiji_coords(sample_info, out_dir = spe_input_dir)

#    A "low resolution" stitched image was produced, which has 1000
#    pixels in its largest dimension
this_image <- imager::load.image(
    file.path(spe_input_dir, "Br2719", "tissue_lowres_image.png")
)
dim(this_image)
library("imager")
plot(this_image)

#    'prep_fiji_image' produced an image and scalefactors
out_paths_image

#    'prep_fiji_coords' produced a file of spatial coordinates for the
#    stitched Br2719
readr::read_csv(out_path_coords)
}
\author{
Nicholas J. Eagles
}
