% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/motif_peaks.R
\name{motif_peaks}
\alias{motif_peaks}
\title{Look for overrepresented motif position peaks in a set of sequences.}
\usage{
motif_peaks(hits, seq.length, seq.count, bandwidth, max.p = 1e-06,
  peak.width = 3, nrand = 100, plot = TRUE, BP = FALSE)
}
\arguments{
\item{hits}{\code{numeric} A vector of sequence positions indicating motif sites.}

\item{seq.length}{\code{numeric(1)} Length of sequences. Only one number is
allowed, as all sequences must be of identical length. If missing,
then the largest number from \code{hits} is used.}

\item{seq.count}{\code{numeric(1)} Number of sequences with motif sites. If
missing, then the number of unique values in \code{hits} is used.}

\item{bandwidth}{\code{numeric(1)} Peak smoothing parameter. Smaller numbers
will result in skinnier peaks, larger numbers will result in wider
peaks. Leaving this empty will cause \code{\link[=motif_peaks]{motif_peaks()}} to generate one
by itself (see 'details').}

\item{max.p}{\code{numeric(1)} Maximum P-value allowed for finding significant
motif site peaks.}

\item{peak.width}{\code{numeric(1)} Minimum peak width. A peak is defined as
as the highest point within the value set by \code{peak.width}.}

\item{nrand}{\code{numeric(1)} Number of random permutations for generating a
null distribution. In order to calculate P-values, a set of random
motif site positions are generated \code{nrand} times.}

\item{plot}{\code{logical(1)} Will create a \code{ggplot2} object displaying motif
peaks.}

\item{BP}{\code{logical(1)} Allows for the use of \pkg{BiocParallel} within
\code{\link[=motif_peaks]{motif_peaks()}}. See \code{\link[BiocParallel:register]{BiocParallel::register()}} to change the
default backend. Setting \code{BP = TRUE} is only recommended for
exceptionally large jobs. Keep in mind that this function will not
attempt to limit its memory usage.}
}
\value{
A \code{DataFrame} with peak positions and P-values. If \code{plot = TRUE},
then a list is returned with the \code{DataFrame} as the first item and
the \code{ggplot2} object as the second item.
}
\description{
Using the motif position data from \code{\link[=scan_sequences]{scan_sequences()}} (or elsewhere),
test whether certain positions in the sequences have significantly higher
motif density.
}
\details{
Kernel smoothing is used to calculate motif position density. The
implementation for this process is based on code from the
\pkg{KernSmooth} R package (Wand 2015). These
density estimates are used to
determine peak locations and heights. To calculate the P-values of
these peaks, a null distribution is calculated from peak heights of
randomly generated motif positions.

If the \code{bandwidth} option is not supplied, then the following code is used
(from \pkg{KernSmooth}):

\code{del0 <- (1 / (4 * pi))^(1 / 10)}

\code{bandwidth <- del0 * (243 / (35 * length(hits)))^(1 / 5) * sqrt(var(hits))}
}
\examples{
data(ArabidopsisMotif)
data(ArabidopsisPromoters)
if (R.Version()$arch != "i386") {
hits <- scan_sequences(ArabidopsisMotif, ArabidopsisPromoters, RC = FALSE)
res <- motif_peaks(as.vector(hits$start), 1000, 50)
# View plot:
res$Plot

# The raw plot data can be found in:
res$Plot$data
}

}
\references{
Wand M (2015). \emph{KernSmooth: Functions for Kernel Smoothing
Supporting Wand and Jones (1995)}. R package version 2.23-15,
<URL: https://CRAN.R-project.org/package=KernSmooth>.
}
\seealso{
\code{\link[=scan_sequences]{scan_sequences()}}
}
\author{
Benjamin Jean-Marie Tremblay, \email{benjamin.tremblay@uwaterloo.ca}
}
