## ----options, include = FALSE-------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

# already load SeuratObject and SingleCellExperiment
# so warnings generated by these packages are not shown
library(SeuratObject)
library(SingleCellExperiment)

## ----install, eval = FALSE----------------------------------------------------
# if (!requireNamespace("BiocManager", quietly = TRUE)) {
#   install.packages("BiocManager")
# }
# BiocManager::install("anndataR")

## ----h5ad-path----------------------------------------------------------------
library(anndataR)

h5ad_path <- system.file("extdata", "example.h5ad", package = "anndataR")

## ----read-in-memory-----------------------------------------------------------
adata <- read_h5ad(h5ad_path)

## ----read-as-SingleCellExperiment---------------------------------------------
sce <- read_h5ad(h5ad_path, as = "SingleCellExperiment")

## ----read-as-Seurat-----------------------------------------------------------
obj <- read_h5ad(h5ad_path, as = "Seurat")

## ----read-on-disk-------------------------------------------------------------
adata <- read_h5ad(h5ad_path, as = "HDF5AnnData")

## ----show-structure-----------------------------------------------------------
adata

## ----access-slots-------------------------------------------------------------
dim(adata$X)
adata$obs[1:5, 1:6]
adata$var[1:5, 1:6]

## ----as-SingleCellExperiment--------------------------------------------------
sce <- adata$as_SingleCellExperiment()
sce

## ----as-Seurat----------------------------------------------------------------
obj <- adata$as_Seurat()
obj

## ----as-AnnData-from-SingleCellExperiment-------------------------------------
adata <- as_AnnData(sce)
adata

## ----as-AnnData-from-Seurat---------------------------------------------------
adata <- as_AnnData(obj)
adata

## ----manually-create-object---------------------------------------------------
adata <- AnnData(
  X = matrix(rnorm(100), nrow = 10),
  obs = data.frame(
    cell_type = factor(rep(c("A", "B"), each = 5))
  ),
  var = data.frame(
    gene_name = paste0("gene_", 1:10)
  )
)

adata

## ----write-to-disk------------------------------------------------------------
tmpfile <- tempfile(fileext = ".h5ad")
adata$write_h5ad(tmpfile) # Alternatively, write_h5ad(adata, tmpfile)

## ----write-SingleCellExperiment-to-disk---------------------------------------
tmpfile <- tempfile(fileext = ".h5ad")
write_h5ad(sce, tmpfile)

## ----write-Seurat-to-disk-----------------------------------------------------
tmpfile <- tempfile(fileext = ".h5ad")
write_h5ad(obj, tmpfile)

## ----subset-obs---------------------------------------------------------------
# Create some sample data
adata <- AnnData(
  X = matrix(rnorm(50), nrow = 10, ncol = 5),
  obs = data.frame(
    cell_type = factor(rep(c("A", "B", "C"), c(3, 4, 3))),
    score = runif(10)
  ),
  var = data.frame(
    gene_name = paste0("gene_", 1:5),
    highly_variable = c(TRUE, FALSE, TRUE, TRUE, FALSE)
  )
)

# Subset to cell type "A"
view1 <- adata[adata$obs$cell_type == "A", ]
view1

## ----subset-vars--------------------------------------------------------------
# Subset to highly variable genes
view2 <- adata[, adata$var$highly_variable]
view2

## ----subset-both--------------------------------------------------------------
# Subset to cell type "A" and highly variable genes
view3 <- adata[adata$obs$cell_type == "A", adata$var$highly_variable]
view3

## ----subset-types-------------------------------------------------------------
# Numeric indices
view4 <- adata[1:5, 1:3]
view4

# Character names (if available)
rownames(adata) <- paste0("cell_", 1:10)
colnames(adata) <- paste0("gene_", 1:5)
view5 <- adata[c("cell_1", "cell_2"), c("gene_1", "gene_3")]
view5

## ----view-properties----------------------------------------------------------
# Access dimensions
dim(view3)
nrow(view3)
ncol(view3)

# Access names
rownames(view3)
colnames(view3)

# Access data matrices and metadata
view3$X
view3$obs
view3$var

# Views can be converted to concrete implementations
concrete <- view3$as_InMemoryAnnData()
concrete

## ----citation-----------------------------------------------------------------
citation("anndataR")

## ----sessioninfo--------------------------------------------------------------
sessionInfo()

