% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ScoreSignatures_UCell.R
\name{ScoreSignatures_UCell}
\alias{ScoreSignatures_UCell}
\title{Calculate module enrichment scores from single-cell data}
\usage{
ScoreSignatures_UCell(
  matrix = NULL,
  features,
  precalc.ranks = NULL,
  maxRank = 1500,
  w_neg = 1,
  name = "_UCell",
  assay = "counts",
  chunk.size = 100,
  missing_genes = c("impute", "skip"),
  BPPARAM = NULL,
  ncores = 1,
  ties.method = "average",
  force.gc = FALSE
)
}
\arguments{
\item{matrix}{Input matrix, either stored in a \link[SingleCellExperiment:SingleCellExperiment]{SingleCellExperiment::SingleCellExperiment} object
or as a raw matrix. \code{dgCMatrix} format supported.}

\item{features}{A list of signatures, for example:
\code{list(Tcell_signature = c("CD2","CD3E","CD3D"),
    Myeloid_signature = c("SPI1","FCER1G","CSF1R"))}
You can also specify positive and negative gene sets by adding a + or -
sign to genes in the signature; see an example below}

\item{precalc.ranks}{If you have pre-calculated ranks using
\code{\link{StoreRankings_UCell}}, you can specify the pre-calculated
ranks instead of the gene vs. cell matrix.}

\item{maxRank}{Maximum number of genes to rank per cell; above this rank, a
given gene is considered as not expressed. Note: this parameter is
ignored if \code{precalc.ranks} are specified}

\item{w_neg}{Weight on negative genes in signature. e.g. \code{w_neg=1} weighs
equally up- and down-regulated genes, \code{w_neg=0.5} gives 50\% less
importance to negative genes}

\item{name}{Name suffix appended to signature names}

\item{assay}{The sce object assay where the data is to be found}

\item{chunk.size}{Number of cells to be processed simultaneously (lower size
requires slightly more computation but reduces memory demands)}

\item{missing_genes}{How to handle missing genes in matrix:
"impute": impute expression to zero; "skip": remove missing
genes from signature}

\item{BPPARAM}{A \code{\link[BiocParallel:register]{BiocParallel::bpparam()}} object that tells UCell
how to parallelize. If provided, it overrides the \code{ncores} parameter.}

\item{ncores}{Number of processors to parallelize computation. If
\code{BPPARAM = NULL}, the function uses
\code{BiocParallel::MulticoreParam(workers=ncores)}}

\item{ties.method}{How ranking ties should be resolved - passed on to
\link[data.table:frank]{data.table::frank}}

\item{force.gc}{Explicitly call garbage collector to reduce memory footprint}
}
\value{
Returns input SingleCellExperiment object with UCell scores
added to altExp
}
\description{
Given a gene vs. cell matrix, calculates module/signature enrichment scores
on single-cell level using Mann-Whitney U statistic.
UCell scores are normalized U statistics (between 0 and 1), and they are
mathematically related to the Area under the ROC curve (see
\href{https://doi.org/10.1256/003590002320603584}{Mason and Graham})
These scores only depend on the gene expression ranks of individual cell,
and therefore they are robust across datasets regardless of dataset
composition.
}
\examples{
library(UCell)
# Using sparse matrix
data(sample.matrix)
gene.sets <- list( Tcell_signature = c("CD2","CD3E","CD3D"),
                 Myeloid_signature = c("SPI1","FCER1G","CSF1R"))
scores <- ScoreSignatures_UCell(sample.matrix, features=gene.sets)
head(scores)

# Using sce object
library(SingleCellExperiment)
data(sample.matrix)
my.sce <- SingleCellExperiment(list(counts=sample.matrix))
gene.sets <- list( Tcell_signature = c("CD2","CD3E","CD3D"),
                 Myeloid_signature = c("SPI1","FCER1G","CSF1R"))
my.sce <- ScoreSignatures_UCell(my.sce, features=gene.sets)
altExp(my.sce, 'UCell')

}
