% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/perform_and_visualize_DA.R
\name{perform_and_visualize_DA}
\alias{perform_and_visualize_DA}
\title{Perform and Visualize Differential Abundance Analysis with edgeR or DESeq2}
\usage{
perform_and_visualize_DA(
  obj,
  method,
  group_var = NULL,
  contrast,
  pseudocount = 1,
  significance_level = 0.05,
  output_csv_path = NULL,
  target_glom = "Genus",
  palette = c("#FFEB3B", "#073B4C"),
  global_fdr = TRUE
)
}
\arguments{
\item{obj}{A \code{phyloseq} or \code{TreeSummarizedExperiment} object.}

\item{method}{A string: either \code{"edgeR"} or \code{"DESeq2"}.}

\item{group_var}{A string specifying the grouping variable in the sample metadata.
Not required if \code{contrast} is a named list for multiple variables.}

\item{contrast}{One of the following:
\itemize{
\item A character vector of two levels to compare (e.g., \code{c("Control", "Treated")}).
\item A list of such character vectors for multiple contrasts within one grouping variable.
\item A named list of such lists (e.g., \code{list(Treatment = list(c("A", "B"), c("A", "C")), Genotype = list(...))}).
}}

\item{pseudocount}{A numeric value used to replace zero or negative counts (default = \code{1}).}

\item{significance_level}{A numeric value for the FDR threshold to determine significance (default = \code{0.05}).}

\item{output_csv_path}{Optional path to save results as CSV files. For multiple contrasts, each is saved separately.}

\item{target_glom}{A string specifying the taxonomic rank to aggregate taxa (default = \code{"Genus"}).}

\item{palette}{A vector of colors for plotting significant and non-significant points (default = \code{c("#FFEB3B", "#073B4C")}).}

\item{global_fdr}{Logical. If \code{TRUE}, applies FDR correction across all contrasts (default = \code{FALSE}).}
}
\value{
A list of result objects, or a single result if a single contrast is given. Each result includes:
\item{results}{A \code{data.frame} of differential abundance results with taxonomic annotations.}
\item{obj_significant}{A filtered \code{phyloseq} or \code{TreeSummarizedExperiment} object with significant taxa.}
\item{plot}{A \code{ggplot2} volcano plot.}
\item{bar_plot}{A \code{ggplot2} bar plot of log fold changes and standard errors.}
\item{bar_abundance_plot}{A \code{ggplot2} bar plot showing relative abundance of significant taxa across groups.}
}
\description{
Performs differential abundance analysis using edgeR or DESeq2, and visualizes results
with a volcano plot, a log-fold change bar plot, and a relative abundance bar plot
for significantly enriched taxa. Supports comparisons across one or more variables and
allows global FDR correction across multiple contrasts.
}
\details{
For edgeR, the standard error of log-fold change (lfcSE) is estimated using the formula:
\code{lfcSE = logFC / sqrt(LR)}, based on the likelihood ratio test statistic.

When providing multiple contrasts, FDR correction can be applied globally using the
\code{global_fdr = TRUE} option.
}
\examples{
\donttest{
if (requireNamespace("phyloseq", quietly = TRUE)) {
  data("physeq_16SOTU", package = "DspikeIn")

  # Salamander samples belonging to two diet types
  ps_sal <- phyloseq::subset_samples(
    physeq_16SOTU,
    Animal.type == "Salamander" &
      Diet \%in\% c("Insectivore", "Carnivore")
  )

  # Remove taxa with zero counts 
  ps_sal <- phyloseq::prune_taxa(
    phyloseq::taxa_sums(ps_sal) > 0,
    ps_sal
  )

  # Differential abundance test at Genus level
  da_res <- perform_and_visualize_DA(
    obj = ps_sal,
    method = "DESeq2",
    group_var = "Diet",
    significance_level = 0.05,
    contrast = c("Insectivore", "Carnivore"),
    target_glom = "Genus"
  )

  # Visualize results
  if (!is.null(da_res$plot)) print(da_res$plot)
  head(da_res$results)

  # Example: multiple contrasts (optional demonstration)
  contrast_list <- list(c("Insectivore", "Carnivore"))
  da_multi <- perform_and_visualize_DA(
    obj = ps_sal,
    method = "DESeq2",
    group_var = "Diet",
    significance_level = 0.01,
    contrast = contrast_list,
    target_glom = "Genus",
    global_fdr = TRUE
  )

  if (!is.null(da_multi[[1]]$bar_plot))
    print(da_multi[[1]]$bar_plot)
}
}

}
