### R code from vignette source 'Annotation.Rnw'

###################################################
### code chunk number 1: setup
###################################################
options(width=90)
library(useR2013)


###################################################
### code chunk number 2: select
###################################################
cols(org.Dm.eg.db)
keytypes(org.Dm.eg.db)
uniprotKeys <- head(keys(org.Dm.eg.db, keytype="UNIPROT"))
cols <- c("SYMBOL", "PATH")
select(org.Dm.eg.db, keys=uniprotKeys, cols=cols, keytype="UNIPROT")


###################################################
### code chunk number 3: select-kegg
###################################################
kegg <- select(org.Dm.eg.db, "00310", c("UNIPROT", "SYMBOL"), "PATH")
nrow(kegg)
head(kegg, 3)


###################################################
### code chunk number 4: chipseq-anno-data
###################################################
stamFile <- system.file("data", "stam.Rda", package="useR2013")
load(stamFile)


###################################################
### code chunk number 5: chipseq-anno-common
###################################################
ridx <- rowSums(assays(stam)[["Tags"]] > 0) == ncol(stam)
peak <- rowData(stam)[ridx]


###################################################
### code chunk number 6: chipseq-anno-centers
###################################################
peak <- resize(peak, width=1, fix="center")


###################################################
### code chunk number 7: chipseq-anno-tss
###################################################
library(TxDb.Hsapiens.UCSC.hg19.knownGene)
tx <- transcripts(TxDb.Hsapiens.UCSC.hg19.knownGene)
tss <- resize(tx, width=1)


###################################################
### code chunk number 8: chipseq-anno-tss-dist
###################################################
idx <- nearest(peak, tss)
sgn <- as.integer(ifelse(strand(tss)[idx] == "+", 1, -1))
dist <- (start(peak) - start(tss)[idx]) * sgn


###################################################
### code chunk number 9: chipseq-anno-tss-dist
###################################################
bound <- 1000
ok <- abs(dist) < bound
dist <- dist[ok]
table(sign(dist))


###################################################
### code chunk number 10: anno-tss-disthist
###################################################
griddensityplot <-
    function(...)
    ## 'panel' function to plot a grid underneath density
{
    panel.grid()
    panel.densityplot(...)
}
print(densityplot(dist[ok], plot.points=FALSE,
    panel=griddensityplot,
    xlab="Distance to Nearest TSS"))


###################################################
### code chunk number 11: tss-dist-func
###################################################
distToTss <-
    function(peak, tx)
{
    peak <- resize(peak, width=1, fix="center")
    tss <- resize(tx, width=1)
    idx <- nearest(peak, tss)
    sgn <- as.numeric(ifelse(strand(tss)[idx] == "+", 1, -1))
    (start(peak) - start(tss)[idx]) * sgn
}


###################################################
### code chunk number 12: chipseq-anno-seq
###################################################
library(BSgenome.Hsapiens.UCSC.hg19)
ridx <- rowSums(assays(stam)[["Tags"]] > 0) == ncol(stam)
ridx <- ridx & (seqnames(rowData(stam)) == "chr6")
pk6 <- rowData(stam)[ridx]
seqs <- getSeq(Hsapiens, pk6)
head(seqs, 3)


###################################################
### code chunk number 13: chipseq-tss-dist-2
###################################################
pwm <- useR2013::getJASPAR("MA0139.1")
hits <- lapply(seqs, matchPWM, pwm=pwm)
hasPwmMatch <- sapply(hits, length) > 0
dist <- distToTss(pk6, tx)

ok <- abs(dist) < bound
df <- data.frame(Distance = dist[ok], HasPwmMatch = hasPwmMatch[ok])
print(densityplot(~Distance, group=HasPwmMatch, df,
    plot.points=FALSE, panel=griddensityplot,
    auto.key=list(
      columns=2,
      title="Has Position Weight Matrix?",
      cex.title=1),
    xlab="Distance to Nearest Tss"))


###################################################
### code chunk number 14: readVcf
###################################################
library(VariantAnnotation)
fl <- system.file("extdata", "chr22.vcf.gz",
                  package="VariantAnnotation")
(hdr <- scanVcfHeader(fl))
info(hdr)[c("VT", "RSQ"),]


###################################################
### code chunk number 15: readVcf
###################################################
(vcf <- readVcf(fl, "hg19"))
head(rowData(vcf), 3)


###################################################
### code chunk number 16: renameSeqlevels
###################################################
rowData(vcf) <- renameSeqlevels(rowData(vcf), c("22"="ch22"))


###################################################
### code chunk number 17: dbSNP
###################################################
library(SNPlocs.Hsapiens.dbSNP.20101109)
snpFilt <- useR2013::dbSNPFilter("SNPlocs.Hsapiens.dbSNP.20101109")
inDbSNP <- snpFilt(vcf)
table(inDbSNP)


###################################################
### code chunk number 18: SNP-quality
###################################################
metrics <-
    data.frame(inDbSNP=inDbSNP, RSQ=info(vcf)$RSQ)


###################################################
### code chunk number 19: RSQ-plot
###################################################
library(ggplot2)
ggplot(metrics, aes(RSQ, fill=inDbSNP)) +
    geom_density(alpha=0.5) +
    scale_x_continuous(name="MaCH / Thunder Imputation Quality") +
    scale_y_continuous(name="Density") +
    theme(legend.position="top")


