### Extract the gene ranges for Yeast as a RangedData object.
### Note: would be easy to adapt to make it work for any organism for which
### an org.* package is available.
extractYeastGenesAsRangedData <- function()
{
    library(org.Sc.sgd.db)
    chrloc <- toTable(org.Sc.sgdCHRLOC)
    chrlocend <- toTable(org.Sc.sgdCHRLOCEND)
    ## Sanity check (checking that the systematic name, chromosome and strand
    ## are the same in both data frames).
    if (!identical(chrloc[-2], chrlocend[-2])
     || !identical(chrloc[[2]] > 0, chrlocend[[2]] > 0))
        stop("inconsistent CHRLOC/CHRLOCEND maps for Yeast")
    gene_starts <- abs(chrloc[[2]])
    gene_ends <- abs(chrlocend[[2]])
    # Filter out autonomously replicating sequences.
    ok <- gene_starts <= gene_ends
    gene_ranges <- IRanges(start=gene_starts[ok], end=gene_ends[ok])
    RangedData(gene_ranges, space=chrloc[[3]][ok],
               strand=strand(ifelse(chrloc[[2]][ok] > 0, "+", "-")),
               systematic_name=chrloc[[1]][ok])
}

### Extract the gene ranges for Yeast as a RangesList object with 1 element
### per chromosome. Use the 'strand' arg. to only extract genes that belong
### to the specified strand.
### Note: would be easy to adapt to make it work for any organism for which
### an org.* package is available.
extractYeastGenesAsRangesList <- function(strand=NA)
{
    library(org.Sc.sgd.db)
    chrloc <- toTable(org.Sc.sgdCHRLOC)
    chrlocend <- toTable(org.Sc.sgdCHRLOCEND)
    ## Sanity check (checking that the systematic name, chromosome and strand
    ## are the same in both data frames).
    if (!identical(chrloc[-2], chrlocend[-2])
     || !identical(chrloc[[2]] > 0, chrlocend[[2]] > 0))
        stop("inconsistent CHRLOC/CHRLOCEND maps for Yeast")
    if (!is.na(strand)) {
        if (strand == "+") {
            chrloc <- chrloc[chrloc[[2]] > 0, ]
            chrlocend <- chrlocend[chrlocend[[2]] > 0, ]
        } else if (strand == "-") {
            chrloc <- chrloc[chrloc[[2]] < 0, ]
            chrlocend <- chrlocend[chrlocend[[2]] < 0, ]
        }
    }
    gene_starts <- abs(chrloc[[2]])
    gene_ends <- abs(chrlocend[[2]])
    # Filter out autonomously replicating sequences.
    ok <- gene_starts <= gene_ends
    gene_ranges <- IRanges(start=gene_starts[ok], end=gene_ends[ok],
                           names=chrloc[[1]][ok])
    split(gene_ranges, chrloc[[3]][ok])
}

### Will work on any result of the 2 previous functions.
normalizeGeneChromosome <- function(x)
{
    chrom <- paste("chr", names(x), sep="")
    chrom[grep("chr17", chrom)] <- "chrmt"
    names(x) <- chrom
    x
}

